/** @file   grenade.cpp
 * @brief   Implementation of Grenade - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "grenade.h"
#include "soundsamples.h"
#include "GfxManager.h"
#include "www_map.h"
#include "bullettable.h"
#include "animplayer.h"
#include "gameanims.h"
#include "GfxId.h"
#include "AnimId.h"
#include "Settings.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
Grenade::Grenade( GameObject* aOwner ) :
  Ammunition( aOwner ),
  
  iType( EDetonatable ),
  iLifetime( 100 ),
  iFragmentCount( 32 ),
  iDamage( 10 ),
  iDetonated( false )
{
}



/** Destructor
 */
Grenade::~Grenade()
{
}




///
/// Public methods
/// ==============

/** Updates this grenade
 */
bool Grenade::updateAmmunition()
{
  // If our movement speed is too slow, we stop it
  if ( iVelocity.length() < 1 ) {
    iVelocity.set( 0,0 );
    iVelocityVectorLength = 0;
    iAltitude = 0;
    
    
  } else {
    if ( this->iAltitude < 0 ) {
      // We've touched the ground
      iAltitude = 0;
      iVerticalSpeed = - this->iVerticalSpeed;
      iVerticalSpeed *= 0.7;
      
      iVelocity *= 0.5;
      iVelocityVectorLength *= 0.5;
      
      Sound::playSample( SMP_BOUNCE, false );
    }
  }
  
  // Manage the lifetimes
  iLifetime -= 1;
  if ( iLifetime < 0 ) {
    // Time to explode the grenade
    return true;
  }
  return false;
}



/** Draws this grenade
 */
void Grenade::redraw( BITMAP* aTarget ) const
{
  RLE_SPRITE* spriteGrenade = GfxManager::findRleSprite( GfxId::KGrenadeGraphics, 0 );
  RLE_SPRITE* spriteShadow = GfxManager::findRleSprite( GfxId::KGrenadeGraphics, 1 );
  
  Vec2D grenPos( iPosition );
  grenPos -= Vec2D( Map::scrollX, Map::scrollY );
  
  Vec2D shadowPos( grenPos );
  float alt = iAltitude / 5;
  shadowPos += Vec2D( -alt, alt );
  
  
  grenPos -= Vec2D( spriteGrenade->w/2, spriteGrenade->h/2 );
  shadowPos -= Vec2D( spriteShadow->w/2, spriteShadow->h/2 );
  
  draw_rle_sprite( aTarget, spriteShadow, shadowPos.intX(), shadowPos.intY() );
  draw_rle_sprite( aTarget, spriteGrenade, grenPos.intX(), grenPos.intY() );
}



/** Explodes the grenade
 */
void Grenade::endingActions( BulletTable* aBulletTable )
{
  Weapon tmpWeapon( Weapon::W_GRENADE );
  float a = 0;
  float angleStep = 256.0 / static_cast<float>( iFragmentCount );
  for ( int i=0; i < iFragmentCount; i++ ) {
    Bullet* bullet = BulletTable::createBullet( iOwner, iPosition,
                                                Bullet::EGrenade );
    Vec2D veloVec( bullet->velocity() );
    veloVec.rotate( a );
    bullet->setVelocity( veloVec );
    
    aBulletTable->spawnBullet( bullet );
    a += angleStep;
  }
  const Animation lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
  if ( Settings::explosionLights == true ) {
    AnimPlayer::spawn( lightAnim, iPosition, 0 );
  }
  const Animation& lightRing = GameAnims::findAnimation( AnimId::KExplosionLightRing );
  AnimPlayer::spawn( lightRing, iPosition, 0 );
  const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  AnimPlayer::spawn( explosionAnim, iPosition, 0 );
}




/** Detonates this grenade
 */
bool Grenade::detonate( GameObject* aOwner )
{
  if ( aOwner == iOwner || aOwner == 0 ) {
    if ( iDetonated == false ) {
      iLifetime = 5;
      iDetonated = true;
      return true;
    }
  }
  return false;
}



///
/// Getter methods
/// ==============




///
/// Private or Protected methods
/// ============================

} // end of namespace
